<?php
/**
 * Plugin Name: VM Sync Bridge
 * Plugin URI: https://victormisa.com/recursos/
 * Description: Sincroniza contenidos entre dos sitios WordPress (mismo plugin actúa como emisor y receptor). Compatible con Elementor (_elementor_data). Reemplaza dominios de desarrollo → producción. Muestra progreso en la UI.
 * Version: 1.0.0
 * Author: Víctor Misa
 * Author URI: https://victormisa.com
 * Requires at least: 5.8
 * Requires PHP: 7.4
 */
if (!defined('ABSPATH')) { exit; }

// ====== Constantes ======
const VMSYNC_OPT_KEY = 'vmsync_settings';
const VMSYNC_VER     = '1.0.0';

/**
 * Opciones por defecto
 */
function vmsync_default_opts(): array {
	return [
		'remote_base'      => '', // https://cliente.com
		'api_token'        => '', // Secreto compartido
		'auto_push'        => 0,  // 1 = al guardar post
		'post_types'       => 'post,page', // CSV
		'dev_domain'       => '', // dev.local
		'prod_domain'      => '', // www.cliente.com
		'rewrite_enabled'  => 1,  // activar reemplazo
	];
}

/**
 * Carga opciones
 */
function vmsync_opts(): array {
	$opts = get_option(VMSYNC_OPT_KEY, []);
	return wp_parse_args((array)$opts, vmsync_default_opts());
}

// ====== Admin UI ======
add_action('admin_menu', function(){
	add_management_page('VM Sync Bridge', 'VM Sync Bridge', 'manage_options', 'vmsync-bridge', 'vmsync_render_tools');
});

add_action('admin_init', function(){
	register_setting('vmsync_group', VMSYNC_OPT_KEY);

	add_settings_section('vmsync_main', __('Conexión remota', 'vmsync'), '__return_false', 'vmsync-bridge');
	add_settings_field('remote_base', 'URL base del sitio destino', 'vmsync_f_remote_base', 'vmsync-bridge', 'vmsync_main');
	add_settings_field('api_token', 'API Token compartido', 'vmsync_f_api_token', 'vmsync-bridge', 'vmsync_main');
	add_settings_field('post_types', 'Post types a sincronizar (CSV)', 'vmsync_f_post_types', 'vmsync-bridge', 'vmsync_main');
	add_settings_field('auto_push', 'Auto-enviar al guardar', 'vmsync_f_auto_push', 'vmsync-bridge', 'vmsync_main');

	add_settings_section('vmsync_rewrite', __('Reemplazo de dominios', 'vmsync'), '__return_false', 'vmsync-bridge');
	add_settings_field('rewrite_enabled', 'Activar reemplazo', 'vmsync_f_rewrite_enabled', 'vmsync-bridge', 'vmsync_rewrite');
	add_settings_field('dev_domain', 'Dominio de desarrollo', 'vmsync_f_dev_domain', 'vmsync-bridge', 'vmsync_rewrite');
	add_settings_field('prod_domain', 'Dominio del cliente', 'vmsync_f_prod_domain', 'vmsync-bridge', 'vmsync_rewrite');
});

function vmsync_field_input($key, $type='text', $attrs=''){
	$opts = vmsync_opts();
	$val  = esc_attr($opts[$key] ?? '');
	echo '<input class="regular-text" type="'.esc_attr($type).'" name="'.esc_attr(VMSYNC_OPT_KEY).'['.esc_attr($key).']" value="'.$val.'" '.$attrs.' />';
}
function vmsync_f_remote_base(){ vmsync_field_input('remote_base'); }
function vmsync_f_api_token(){ vmsync_field_input('api_token', 'password', 'autocomplete="new-password"'); }
function vmsync_f_post_types(){ vmsync_field_input('post_types'); echo '<p class="description">Ej: <code>post,page,product</code></p>'; }
function vmsync_f_auto_push(){
	$opts = vmsync_opts();
	$chk = !empty($opts['auto_push']) ? 'checked' : '';
	echo '<label><input type="checkbox" name="'.esc_attr(VMSYNC_OPT_KEY).'[auto_push]" value="1" '.$chk.' /> Enviar automáticamente al guardar</label>';
}
function vmsync_f_rewrite_enabled(){
	$opts = vmsync_opts(); $chk = !empty($opts['rewrite_enabled']) ? 'checked' : '';
	echo '<label><input type="checkbox" name="'.esc_attr(VMSYNC_OPT_KEY).'[rewrite_enabled]" value="1" '.$chk.' /> Activar reemplazo de URLs y enlaces internos</label>';
}
function vmsync_f_dev_domain(){ vmsync_field_input('dev_domain'); echo '<p class="description">Ej: <code>dev.midominio.com</code> (sin barra final)</p>'; }
function vmsync_f_prod_domain(){ vmsync_field_input('prod_domain'); echo '<p class="description">Ej: <code>www.cliente.com</code> (sin barra final)</p>'; }

function vmsync_render_tools(){
	if (!current_user_can('manage_options')) return;
	?>
	<div class="wrap">
		<h1>VM Sync Bridge</h1>
		<p>Conecta dos instalaciones WordPress para enviar/recibir contenidos. Este plugin expone un endpoint receptor y permite enviar manual o automáticamente.</p>

		<form method="post" action="options.php">
			<?php settings_fields('vmsync_group'); do_settings_sections('vmsync-bridge'); submit_button(__('Guardar ajustes', 'vmsync')); ?>
		</form>

		<hr/>
		<h2>Herramientas de sincronización</h2>
		<p>
			<button id="vmsync-test" class="button">🔌 Probar conexión (ping)</button>
			<button id="vmsync-push-recent" class="button button-primary">⬆️ Enviar modificados (24h)</button>
		</p>
		<p>
			<label>IDs/Slugs (coma): <input type="text" id="vmsync-idlist" class="regular-text" placeholder="12, 34, post-slug"/></label>
			<button id="vmsync-push-selected" class="button">⬆️ Enviar seleccionados</button>
		</p>
		<div id="vmsync-progress" style="max-width:900px;background:#111;color:#d0ffd0;padding:12px;border-radius:6px;min-height:120px;overflow:auto;font-family:monospace"></div>
	</div>
	<?php
}

add_action('admin_enqueue_scripts', function($hook){
	if ($hook !== 'tools_page_vmsync-bridge') return;
	wp_enqueue_script('vmsync-admin', plugin_dir_url(__FILE__).'vmsync-admin.js', ['jquery'], VMSYNC_VER, true);
	wp_localize_script('vmsync-admin', 'VMSYNC',[
		'ajax' => admin_url('admin-ajax.php'),
		'nonce'=> wp_create_nonce('vmsync_do'),
	]);
});

// JS embebido (sin archivo físico) para simplicidad
add_action('admin_footer', function(){
	$screen = get_current_screen();
	if (!$screen || $screen->id !== 'tools_page_vmsync-bridge') return;
	?>
	<script>
	(function($){
		const $log = $('#vmsync-progress');
		function log(msg){
			const time = new Date().toLocaleTimeString();
			$log.append($('<div/>').text('['+time+'] '+msg));
			$log.scrollTop($log[0].scrollHeight);
		}
		function call(action, data){
			log('Procesando: '+action+'…');
			return $.post(VMSYNC.ajax, Object.assign({action:'vmsync_'+action, _wpnonce:VMSYNC.nonce}, data||{}));
		}
		$('#vmsync-test').on('click', function(e){e.preventDefault(); call('ping').done(r=>{log(r.message||r);}).fail(()=>log('Error en ping'));});
		$('#vmsync-push-recent').on('click', function(e){e.preventDefault(); call('push_recent').done(r=>{(r.logs||[]).forEach(log); log('Hecho.');}).fail(()=>log('Error enviando recientes'));});
		$('#vmsync-push-selected').on('click', function(e){e.preventDefault(); const list=$('#vmsync-idlist').val(); call('push_selected',{list}).done(r=>{(r.logs||[]).forEach(log); log('Hecho.');}).fail(()=>log('Error enviando seleccionados'));});
	})(jQuery);
	</script>
	<?php
});

// ====== AJAX actions (procesamiento con logs) ======
add_action('wp_ajax_vmsync_ping', function(){
	check_ajax_referer('vmsync_do');
	$ok = vmsync_remote_ping();
	wp_send_json(['ok'=>$ok, 'message'=> $ok?'Conexión OK. Endpoint receptor activo.':'Fallo en la conexión o token. Revisa ajustes.']);
});

add_action('wp_ajax_vmsync_push_recent', function(){
	check_ajax_referer('vmsync_do');
	$logs = vmsync_push_recent_posts();
	wp_send_json(['logs'=>$logs]);
});

add_action('wp_ajax_vmsync_push_selected', function(){
	check_ajax_referer('vmsync_do');
	$list = sanitize_text_field($_POST['list'] ?? '');
	$ids  = array_filter(array_map('trim', explode(',', $list)));
	$logs = vmsync_push_id_or_slug_list($ids);
	wp_send_json(['logs'=>$logs]);
});

// ====== Auto push al guardar ======
add_action('save_post', function($post_id, $post, $update){
	$opts = vmsync_opts();
	if (empty($opts['auto_push'])) return;
	$allowed = array_map('trim', explode(',', (string)$opts['post_types']));
	if (!in_array($post->post_type, $allowed, true)) return;
	if (wp_is_post_revision($post_id) || $post->post_status !== 'publish') return;
	// Envío no bloqueante
	vmsync_send_post_async($post_id);
}, 10, 3);

// ====== REST receptor ======
add_action('rest_api_init', function(){
	register_rest_route('vmsync/v1', '/receive', [
		'methods'  => 'POST',
		'permission_callback' => 'vmsync_check_bearer',
		'callback' => 'vmsync_receive_post',
	]);
	register_rest_route('vmsync/v1', '/ping', [
		'methods'  => 'GET',
		'permission_callback' => 'vmsync_check_bearer_optional',
		'callback' => function(){ return new WP_REST_Response(['ok'=>true,'site'=>home_url('/')],200); },
	]);
});

function vmsync_get_auth_header(): string {
	$h = (string)($_SERVER['HTTP_AUTHORIZATION'] ?? '');
	if (!$h && function_exists('apache_request_headers')){
		$hdrs = apache_request_headers();
		$h = $hdrs['Authorization'] ?? '';
	}
	return $h;
}

function vmsync_check_bearer(WP_REST_Request $req){
	$opts = vmsync_opts();
	$auth = vmsync_get_auth_header();
	if (preg_match('/Bearer\s+(.*)/i', $auth, $m)){
		return hash_equals((string)$opts['api_token'], (string)$m[1]);
	}
	return false;
}
function vmsync_check_bearer_optional(){ return true; }

/**
 * Receptor: crea/actualiza por slug o ID
 */
function vmsync_receive_post(WP_REST_Request $req){
	$in = $req->get_json_params();
	if (!is_array($in)) return new WP_REST_Response(['error'=>'bad_json'], 400);

	$post_type = sanitize_key($in['post_type'] ?? 'post');
	$slug      = sanitize_title($in['slug'] ?? '');
	$title     = wp_kses_post($in['title'] ?? '');
	$content   = $in['content'] ?? '';
	$excerpt   = $in['excerpt'] ?? '';
	$status    = sanitize_key($in['status'] ?? 'publish');
	$meta      = is_array($in['meta'] ?? null) ? $in['meta'] : [];
	$tax       = is_array($in['tax'] ?? null) ? $in['tax'] : [];
	$author    = (int)($in['author_id'] ?? 0);
	$post_id   = 0;

	if (empty($slug)) return new WP_REST_Response(['error'=>'missing_slug'], 422);

	$existing = get_page_by_path($slug, OBJECT, $post_type);
	if ($existing){ $post_id = (int)$existing->ID; }

	// Reemplazo de dominios si procede
	list($content, $meta) = vmsync_apply_replacements($content, $meta);

	$payload = [
		'post_type'    => $post_type,
		'post_name'    => $slug,
		'post_title'   => $title,
		'post_content' => wp_kses_post($content),
		'post_excerpt' => wp_kses_post($excerpt),
		'post_status'  => $status,
	];
	if ($author > 0) { $payload['post_author'] = $author; }

	if ($post_id){
		$payload['ID'] = $post_id;
		$post_id = wp_update_post($payload, true);
	} else {
		$post_id = wp_insert_post($payload, true);
	}
	if (is_wp_error($post_id)){
		return new WP_REST_Response(['error'=>'wp_error','message'=>$post_id->get_error_message()], 500);
	}

	// Meta (incluido Elementor)
	foreach ($meta as $k => $v){
		update_post_meta($post_id, sanitize_key($k), vmsync_sanitize_meta_value($k, $v));
	}

	// Taxonomías
	foreach ($tax as $tax_name => $terms){
		$tax_name = sanitize_key($tax_name);
		if (!taxonomy_exists($tax_name)) continue;
		$term_ids = [];
		foreach ((array)$terms as $t){
			if (is_numeric($t)) { $term_ids[] = (int)$t; continue; }
			$term = term_exists((string)$t, $tax_name);
			if (!$term){ $term = wp_insert_term((string)$t, $tax_name); }
			if (!is_wp_error($term)){
				$term_ids[] = (int)($term['term_id'] ?? 0);
			}
		}
		if ($term_ids){ wp_set_post_terms($post_id, $term_ids, $tax_name, false); }
	}

	return new WP_REST_Response(['ok'=>true,'id'=>$post_id], 200);
}

/** Sanitiza meta; _elementor_data permite JSON string */
function vmsync_sanitize_meta_value(string $key, $val){
	if ($key === '_elementor_data'){
		if (is_array($val)) return wp_json_encode($val);
		return (string)$val; // Elementor espera string JSON
	}
	if (is_array($val)) return array_map('sanitize_text_field', $val);
	if (is_string($val)) return wp_kses_post($val);
	return $val;
}

/** Reemplazo de dominios en contenido y meta relevantes */
function vmsync_apply_replacements($content, array $meta): array {
	$opts = vmsync_opts();
	if (empty($opts['rewrite_enabled'])) return [$content, $meta];
	$from = trim((string)$opts['dev_domain']);
	$to   = trim((string)$opts['prod_domain']);
	if ($from && $to){
		$search = [ 'https://'.$from, 'http://'.$from, $from ];
		$replace= [ 'https://'.$to,   'http://'.$to,   $to   ];
		if (is_string($content)){
			$content = str_replace($search, $replace, $content);
		}
		// Meta comunes de Elementor
		foreach (['_elementor_data','_elementor_css','_elementor_edit_mode','_wp_attached_file'] as $mk){
			if (!isset($meta[$mk])) continue;
			if (is_array($meta[$mk])){
				$meta[$mk] = json_decode(str_replace($search, $replace, wp_json_encode($meta[$mk])), true);
			} else {
				$meta[$mk] = str_replace($search, $replace, (string)$meta[$mk]);
			}
		}
	}
	return [$content, $meta];
}

// ====== Envío ======
function vmsync_collect_post_payload(int $post_id): ?array {
	$post = get_post($post_id);
	if (!$post) return null;
	$meta = get_post_meta($post_id);
	// Flatea meta (primera posición)
	$flat = [];
	foreach ($meta as $k => $vals){ $flat[$k] = is_array($vals) ? (count($vals)===1 ? $vals[0] : $vals) : $vals; }

	$payload = [
		'post_type' => $post->post_type,
		'slug'      => $post->post_name,
		'title'     => get_the_title($post),
		'content'   => $post->post_content,
		'excerpt'   => $post->post_excerpt,
		'author_id' => (int)$post->post_author,
		'status'    => $post->post_status,
		'meta'      => [
			'_elementor_data'      => $flat['_elementor_data']      ?? null,
			'_elementor_css'       => $flat['_elementor_css']       ?? null,
			'_elementor_edit_mode' => $flat['_elementor_edit_mode'] ?? null,
		],
		'tax'       => [],
	];
	// Tax: categorías y etiquetas básicas
	$cats = wp_get_post_terms($post_id, 'category', ['fields'=>'names']);
	$tags = wp_get_post_terms($post_id, 'post_tag', ['fields'=>'names']);
	if (!is_wp_error($cats) && $cats){ $payload['tax']['category'] = $cats; }
	if (!is_wp_error($tags) && $tags){ $payload['tax']['post_tag'] = $tags; }
	return $payload;
}

function vmsync_remote_url(string $path): string {
	$opts = vmsync_opts();
	$base = rtrim((string)$opts['remote_base'], '/');
	return $base . '/wp-json' . $path;
}

function vmsync_remote_post(array $payload){
	$opts = vmsync_opts();
	$resp = wp_remote_post(vmsync_remote_url('/vmsync/v1/receive'), [
		'timeout' => 20,
		'headers' => [
			'Authorization' => 'Bearer '.$opts['api_token'],
			'Content-Type'  => 'application/json',
		],
		'body' => wp_json_encode($payload),
	]);
	return $resp;
}

function vmsync_remote_ping(): bool {
	$opts = vmsync_opts();
	if (empty($opts['remote_base']) || empty($opts['api_token'])) return false;
	$resp = wp_remote_get(vmsync_remote_url('/vmsync/v1/ping'), [
		'timeout'=>10,
		'headers'=>['Authorization'=>'Bearer '.$opts['api_token']],
	]);
	if (is_wp_error($resp)) return false;
	$code = wp_remote_retrieve_response_code($resp);
	return $code===200;
}

function vmsync_send_post_async(int $post_id){
	$payload = vmsync_collect_post_payload($post_id);
	if (!$payload) return;
	// No bloquear: disparo inmediato, ignorar resultado
	vmsync_remote_post($payload);
}

function vmsync_push_recent_posts(): array {
	$logs = [];
	$opts = vmsync_opts();
	$types = array_map('trim', explode(',', (string)$opts['post_types']));
	$since = gmdate('Y-m-d H:i:s', time()-DAY_IN_SECONDS);

	$q = new WP_Query([
		'post_type'      => $types,
		'post_status'    => 'publish',
		'posts_per_page' => 50,
		'date_query'     => [['column'=>'post_modified_gmt','after'=> $since ]],
		'orderby'        => 'modified',
		'order'          => 'DESC'
	]);
	if (!$q->have_posts()){ $logs[] = 'No hay posts modificados en 24h.'; return $logs; }
	while ($q->have_posts()){ $q->the_post();
		$id = get_the_ID(); $slug = get_post_field('post_name',$id);
		$payload = vmsync_collect_post_payload($id);
		if (!$payload){ $logs[] = "#{$id} sin payload"; continue; }
		$resp = vmsync_remote_post($payload);
		if (is_wp_error($resp)){
			$logs[] = "#{$id} / {$slug} — ERROR: ".$resp->get_error_message();
		}else{
			$code = wp_remote_retrieve_response_code($resp);
			$body = wp_remote_retrieve_body($resp);
			$logs[] = "#{$id} / {$slug} — HTTP {$code} — {$body}";
		}
	}
	wp_reset_postdata();
	return $logs;
}

function vmsync_push_id_or_slug_list(array $ids_or_slugs): array {
	$logs=[];
	foreach ($ids_or_slugs as $x){
		$post=null;
		if (is_numeric($x)){
			$post = get_post((int)$x);
		}else{
			// Buscar por slug en post y page
			$post = get_page_by_path(sanitize_title($x), OBJECT, ['post','page']);
		}
		if (!$post){ $logs[] = "$x — no encontrado"; continue; }
		$payload = vmsync_collect_post_payload($post->ID);
		$resp = vmsync_remote_post($payload);
		if (is_wp_error($resp)){
			$logs[] = "$x — ERROR: ".$resp->get_error_message();
		}else{
			$logs[] = "$x — OK: HTTP ".wp_remote_retrieve_response_code($resp);
		}
	}
	return $logs;
}

// ====== Fin ======
